;==================================================================================
; This is Grant Searle's code, modified for use with Small Computer Workshop IDE.
;
; SCC  2019-05-04 : v1.0.0  Modified Z80 BIOS for Z180 CPU v1.2
;                           Added support for SC111 CPU with SC119 memory
; SCC  2019-06-27 : v1.0.1  Added support for SC126 motherboard

;==================================================================================
; Contents of this file are copyright Grant Searle
; Blocking/unblocking routines are the published version by Digital Research
; (bugfixed, as found on the web)
;
; You have permission to use this for NON COMMERCIAL USE ONLY
; If you wish to use it elsewhere, please include an acknowledgement to myself.
;
; http://searle.hostei.com/grant/index.html
;
; eMail: home.micros01@btinternet.com
;
; If the above don't work, please perform an Internet search to see if I have
; updated the web page hosting service.
;
;==================================================================================

            .PROC Z180          ;<SCC> SCWorkshop select processor
            .HEXBYTES 0x18      ;<SCC> SCWorkshop Intel Hex output format


; <SCC> Define one of these to select which build is required 
;#DEFINE    S5_Original         ;SC111 Z180Base=0x40 original S5 release
;#DEFINE    SC111_CF64          ;SC111 Z180Base=0xC0 CF=64MB
;#DEFINE    SC111_CF128         ;SC111 Z180Base=0xC0 CF=128MB
;#DEFINE    SC126_CF64          ;SC111 Z180Base=0xC0 CF=64MB
#DEFINE     SC126_CF128         ;SC111 Z180Base=0xC0 CF=128MB
;
; Version         Bus   MHz      I/O    Description
; =======         ===   ===      ===    ===========
; S5_Original     RC    18.432   0x40   SCM S5
; SC111_CF64      RC    18.432   0xC0   SC111 CF64
; SC111_CF128     RC    18.432   0xC0   SC111 CF128
; SC126_CF64      RC    18.432   0xC0   SC126 CF64
; SC126_CF128     RC    18.432   0xC0   SC126 CF128


; <JL> Select one of the two size options: 64MB or 128MB
;#DEFINE    SIZE64
;#DEFINE    SIZE128
#IFDEF      SC111_CF64
#DEFINE     SIZE64
#ENDIF
#IFDEF      SC111_CF128
#DEFINE     SIZE128
#ENDIF
#IFDEF      SC126_CF64
#DEFINE     SIZE64
#ENDIF
#IFDEF      SC126_CF128
#DEFINE     SIZE128
#ENDIF

ccp         .EQU 0D000h         ; Base of CCP.
bdos        .EQU ccp + 0806h    ; Base of BDOS.
bios        .EQU ccp + 1600h    ; Base of BIOS.

; Set CP/M low memory datA, vector and buffer addresses.

iobyte      .EQU 03h            ; Intel standard I/O definition byte.
userdrv     .EQU 04h            ; Current user number and drive.
tpabuf      .EQU 80h            ; Default I/O buffer and command line storage.

; **********************************************************************
; **  Z180 register constants
; **********************************************************************

#IFDEF      S5_Original
kZ180:      .EQU 0x40           ;Z180 internal register base address
#ELSE
kZ180:      .EQU 0xC0           ;Z180 internal register base address
#ENDIF

CNTLA0:     .EQU kZ180 + 0x00   ;ASCI Channel Control Register A chan 0
CNTLA1:     .EQU kZ180 + 0x01   ;ASCI Channel Control Register A chan 1
CNTLB0:     .EQU kZ180 + 0x02   ;ASCI Channel Control Register B chan 0
CNTLB1:     .EQU kZ180 + 0x03   ;ASCI Channel Control Register B chan 1
STAT0:      .EQU kZ180 + 0x04   ;ASCI Status Register channel 0
STAT1:      .EQU kZ180 + 0x05   ;ASCI Status Register channel 1
TDR0:       .EQU kZ180 + 0x06   ;ASCI Transmit Data Register channel 0
TDR1:       .EQU kZ180 + 0x07   ;ASCI Transmit Data Register channel 1
RDR0:       .EQU kZ180 + 0x08   ;ASCI Receive Register channel 0
RDR1:       .EQU kZ180 + 0x09   ;ASCI Receive Register channel 1
ASCI0:      .EQU kZ180 + 0x12   ;ASCI Extension Control Register 0
ASCI1:      .EQU kZ180 + 0x13   ;ASCI Extension Control Register 1
ASTC0L:     .EQU kZ180 + 0x1A   ;ASCI Time Constant Register ch 0 Low
ASTC0H:     .EQU kZ180 + 0x1B   ;ASCI Time Constant Register ch 0 High
ASTC1L:     .EQU kZ180 + 0x1C   ;ASCI Time Constant Register ch 1 Low
ASTC1H:     .EQU kZ180 + 0x1D   ;ASCI Time Constant Register ch 1 High
CMR:        .EQU kZ180 + 0x1E   ;Clock Multiplier Register
CCR         .EQU kZ180 + 0x1F   ;CPU Control Register
DCNTL:      .EQU kZ180 + 0x32   ;DMA/WAIT Control Register
IL:         .EQU kZ180 + 0x33   ;Interrupt Vector Register
ITC:        .EQU kZ180 + 0x34   ;INT/TRAP Control Register
CBR:        .EQU kZ180 + 0x38   ;MMU Control Base Register
BBR:        .EQU kZ180 + 0x39   ;MMU Bank Base Register
CBAR:       .EQU kZ180 + 0x3A   ;MMU Common/Bank Register
OMCR:       .EQU kZ180 + 0x3E   ;Operation Mode Control Register
ICR:        .EQU kZ180 + 0x3F   ;I/O Control Register


; **********************************************************************
; Serial status register bits

ST_RDRF:    .EQU 7              ;Receive data register Full
ST_TDRE:    .EQU 1              ;Transmit data register empty




int38       .EQU 38H
nmi         .EQU 66H

blksiz      .equ 4096           ;CP/M allocation size
hstsiz      .equ 512            ;host disk sector size
hstspt      .equ 32             ;host disk sectors/trk
hstblk      .equ hstsiz/128     ;CP/M sects/host buff
cpmspt      .equ hstblk * hstspt  ;CP/M sectors/track
secmsk      .equ hstblk-1       ;sector mask
            ;compute sector mask
;secshf                         .equ  2   ;log2(hstblk)

wrall       .equ 0              ;write to allocated
wrdir       .equ 1              ;write to directory
wrual       .equ 2              ;write to unallocated



; CF registers
CF_DATA     .EQU $10
CF_FEATURES .EQU $11
CF_ERROR    .EQU $11
CF_SECCOUNT .EQU $12
CF_SECTOR   .EQU $13
CF_CYL_LOW  .EQU $14
CF_CYL_HI   .EQU $15
CF_HEAD     .EQU $16
CF_STATUS   .EQU $17
CF_COMMAND  .EQU $17
CF_LBA0     .EQU $13
CF_LBA1     .EQU $14
CF_LBA2     .EQU $15
CF_LBA3     .EQU $16

;CF Features
CF_8BIT     .EQU 1
CF_NOCACHE  .EQU 082H
;CF Commands
CF_READ_SEC .EQU 020H
CF_WRITE_SEC                    .EQU 030H
CF_SET_FEAT .EQU  0EFH

LF          .EQU 0AH            ;line feed
FF          .EQU 0CH            ;form feed
CR          .EQU 0DH            ;carriage RETurn

;================================================================================================

            .ORG bios           ; BIOS origin.

;================================================================================================
; BIOS jump table.
;================================================================================================
            JP boot             ;  0 Initialize.
wboote:     JP wboot            ;  1 Warm boot.
            JP const            ;  2 Console status.
            JP conin            ;  3 Console input.
            JP conout           ;  4 Console OUTput.
            JP list             ;  5 List OUTput.
            JP punch            ;  6 punch OUTput.
            JP reader           ;  7 Reader input.
            JP home             ;  8 Home disk.
            JP seldsk           ;  9 Select disk.
            JP settrk           ; 10 Select track.
            JP setsec           ; 11 Select sector.
            JP setdma           ; 12 Set DMA ADDress.
            JP read             ; 13 Read 128 bytes.
            JP write            ; 14 Write 128 bytes.
            JP listst           ; 15 List status.
            JP sectran          ; 16 Sector translate.

;================================================================================================
; Disk parameter headers for disk 0 to 15
;================================================================================================
; <JL> Added IFDEF/ELSE block to select 64/128 MB
dpbase:
            .DW 0000h,0000h,0000h,0000h,dirbuf,dpb0,0000h,alv00
            .DW 0000h,0000h,0000h,0000h,dirbuf,dpb,0000h,alv01
            .DW 0000h,0000h,0000h,0000h,dirbuf,dpb,0000h,alv02
            .DW 0000h,0000h,0000h,0000h,dirbuf,dpb,0000h,alv03
            .DW 0000h,0000h,0000h,0000h,dirbuf,dpb,0000h,alv04
            .DW 0000h,0000h,0000h,0000h,dirbuf,dpb,0000h,alv05
            .DW 0000h,0000h,0000h,0000h,dirbuf,dpb,0000h,alv06
#IFDEF      SIZE64
            .DW 0000h,0000h,0000h,0000h,dirbuf,dpbLast,0000h,alv07
#ELSE
            .DW 0000h,0000h,0000h,0000h,dirbuf,dpb,0000h,alv07
            .DW 0000h,0000h,0000h,0000h,dirbuf,dpb,0000h,alv08
            .DW 0000h,0000h,0000h,0000h,dirbuf,dpb,0000h,alv09
            .DW 0000h,0000h,0000h,0000h,dirbuf,dpb,0000h,alv10
            .DW 0000h,0000h,0000h,0000h,dirbuf,dpb,0000h,alv11
            .DW 0000h,0000h,0000h,0000h,dirbuf,dpb,0000h,alv12
            .DW 0000h,0000h,0000h,0000h,dirbuf,dpb,0000h,alv13
            .DW 0000h,0000h,0000h,0000h,dirbuf,dpb,0000h,alv14
            .DW 0000h,0000h,0000h,0000h,dirbuf,dpbLast,0000h,alv15
#ENDIF
            
; First drive has a reserved track for CP/M
dpb0:
            .DW 128             ;SPT - sectors per track
            .DB 5               ;BSH - block shift factor
            .DB 31              ;BLM - block mask
            .DB 1               ;EXM - Extent mask
            .DW 2043            ; (2047-4) DSM - Storage size (blocks - 1)
            .DW 511             ;DRM - Number of directory entries - 1
            .DB 240             ;AL0 - 1 bit set per directory block
            .DB 0               ;AL1 -            "
            .DW 0               ;CKS - DIR check vector size (DRM+1)/4 (0=fixed disk)
            .DW 1               ;OFF - Reserved tracks

dpb:
            .DW 128             ;SPT - sectors per track
            .DB 5               ;BSH - block shift factor
            .DB 31              ;BLM - block mask
            .DB 1               ;EXM - Extent mask
            .DW 2047            ;DSM - Storage size (blocks - 1)
            .DW 511             ;DRM - Number of directory entries - 1
            .DB 240             ;AL0 - 1 bit set per directory block
            .DB 0               ;AL1 -            "
            .DW 0               ;CKS - DIR check vector size (DRM+1)/4 (0=fixed disk)
            .DW 0               ;OFF - Reserved tracks

; Last drive is smaller because CF is never full 64MB or 128MB
; <JL> Added IFDEF/ELSE block to select 64/128 MB
dpbLast:
            .DW 128             ;SPT - sectors per track
            .DB 5               ;BSH - block shift factor
            .DB 31              ;BLM - block mask
            .DB 1               ;EXM - Extent mask
#IFDEF      SIZE64
            .DW 1279            ;DSM - Storage size (blocks - 1)  ; 1279 = 5MB (for 64MB card)
#ELSE
            .DW 511             ;DSM - Storage size (blocks - 1)  ; 511 = 2MB (for 128MB card)
#ENDIF
            .DW 511             ;DRM - Number of directory entries - 1
            .DB 240             ;AL0 - 1 bit set per directory block
            .DB 0               ;AL1 -            "
            .DW 0               ;CKS - DIR check vector size (DRM+1)/4 (0=fixed disk)
            .DW 0               ;OFF - Reserved tracks

;================================================================================================
; Cold boot
;================================================================================================

boot:
            DI                  ; Disable interrupts.
            LD SP,biosstack     ; Set default stack.

;                               Turn off ROM

            LD A,$01
            OUT ($38),A

; Initialise Z180's serial ports
; Assumed to already be initialised by Small Computer Monitor

            ; Interrupt vector in page FF
            ;LD A,$FF
            ;LD I,A

            CALL printInline
            .DB FF

#IFDEF      S5_Original
            .TEXT "CP/M BIOS 1.4 for Z180 with SCM S5 based on code by G. Searle"
#ENDIF
#IFDEF      SC111_CF64
            .TEXT "Z180 BIOS 1.0 for SC111 CF64  based on code by G. Searle"
#ENDIF
#IFDEF      SC111_CF128
            .TEXT "Z180 BIOS 1.0 for SC111 CF128 based on code by G. Searle"
#ENDIF
#IFDEF      SC126_CF64
            .TEXT "Z180 BIOS 1.0 for SC126 CF64  based on code by G. Searle"
#ENDIF
#IFDEF      SC126_CF128
            .TEXT "Z180 BIOS 1.0 for SC126 CF128 based on code by G. Searle"
#ENDIF
#IFDEF      RC_Z180_Native
            .TEXT "CP/M BIOS 1.3 for Z180-RC based on code by G. Searle"
#ENDIF
            .DB CR,LF
            .DB CR,LF
            .TEXT "CP/M 2.2 "
            .TEXT "Copyright"
            .TEXT " 1979 (c) by Digital Research"
            .DB CR,LF,0

            CALL cfWait
            LD  A,CF_8BIT       ; Set IDE to be 8bit
            OUT (CF_FEATURES),A
            LD A,CF_SET_FEAT
            OUT (CF_COMMAND),A


            CALL cfWait
            LD  A,CF_NOCACHE    ; No write cache
            OUT (CF_FEATURES),A
            LD A,CF_SET_FEAT
            OUT (CF_COMMAND),A

            XOR A               ; Clear I/O & drive bytes.
            LD (userdrv),A

            JP gocpm

;================================================================================================
; Warm boot
;================================================================================================

wboot:
            DI                  ; Disable interrupts.
            LD SP,biosstack     ; Set default stack.



            ; Interrupt vector in page FF
            LD A,$FF
            LD I,A


            LD B,11             ; Number of sectors to reload

            LD A,0
            LD (hstsec),A
            LD HL,ccp
rdSectors:

            CALL cfWait

            LD A,(hstsec)
            OUT  (CF_LBA0),A
            LD A,0
            OUT  (CF_LBA1),A
            OUT  (CF_LBA2),A
            LD A,0E0H
            OUT  (CF_LBA3),A
            LD  A,1
            OUT  (CF_SECCOUNT),A

            PUSH  BC

            CALL  cfWait

            LD  A,CF_READ_SEC
            OUT  (CF_COMMAND),A

            CALL  cfWait

            LD  C,4
rd4secs512:
            LD  B,128
rdByte512:
            in  A,(CF_DATA)
            LD  (HL),A
            iNC  HL
            dec  B
            JR  NZ, rdByte512
            dec  C
            JR  NZ,rd4secs512

            POP  BC

            LD A,(hstsec)
            INC A
            LD (hstsec),A

            djnz rdSectors


;================================================================================================
; Common code for cold and warm boot
;================================================================================================

gocpm:
            xor A               ;0 to accumulator
            ld (hstact),A       ;host buffer inactive
            ld (unacnt),A       ;clear unalloc count

            ;LD HL,serialInt    ; ADDress of serial interrupt.
            ;LD ($40),HL

            LD HL,tpabuf        ; ADDress of BIOS DMA buffer.
            LD (dmaAddr),HL
            LD A,0C3h           ; Opcode for 'JP'.
            LD (00h),A          ; Load at start of RAM.
            LD HL,wboote        ; ADDress of jump for a warm boot.
            LD (01h),HL
            LD (05h),A          ; Opcode for 'JP'.
            LD HL,bdos          ; ADDress of jump for the BDOS.
            LD (06h),HL
            LD A,(userdrv)      ; Save new drive number (0).
            LD C,A              ; Pass drive number in C.

            ;IM 2
            ;EI                 ; Enable interrupts

            JP ccp              ; Start CP/M by jumping to the CCP.

;================================================================================================
; Console I/O routines
;================================================================================================

;------------------------------------------------------------------------------------------------
; Input device status
; Returns its status in A; 0 if no character is ready, 0FFh if one is available.

const:      LD A,(iobyte)
            AND 00001011b       ; Mask off console and high bit of reader
            CP 00001010b        ; redirected to reader on UR1/2 (Serial A)
            JR Z,constA
            CP 00000010b        ; redirected to reader on TTY/RDR (Serial B)
            JR Z,constB

            AND $03             ; remove the reader from the mask - only console bits then remain
            CP $01
            JR NZ,constB

constA:     IN0  A,(STAT0)      ;Read serial port status register
            BIT  ST_RDRF,A      ;Receive register full?
            JR   Z,constNone    ;Skip if no character available
            LD   A,0xFF         ;Character is available
            RET

constB:     IN0  A,(STAT1)      ;Read serial port status register
            BIT  ST_RDRF,A      ;Receive register full?
            JR   Z,constNone    ;Skip if no character available
            LD   A,0xFF         ;Character is available
            RET

constNone:
            LD   A,0            ;Character not available
            RET


;------------------------------------------------------------------------------------------------
; Character input
; Wait until a character is available, then return it in A.

reader:     LD A,(iobyte)
            AND $08
            CP $08
            JR NZ,coninB
            JR coninA

conin:      LD A,(iobyte)
            AND $03
            CP $02
            JR Z,reader         ; "BAT:" redirect
            CP $01
            JR NZ,coninB
            
coninA:     IN0  A,(STAT0)      ;Read serial port status register
            BIT  ST_RDRF,A      ;Receive register full?
            JR   Z, coninA      ;No, so keep waiting
            IN0  A,(RDR0)       ;Read data byte
            RET                 ;Return character in A

coninB:     IN0  A,(STAT1)      ;Read serial port status register
            BIT  ST_RDRF,A      ;Receive register full?
            JR   Z, coninB      ;No, so keep waiting
            IN0  A,(RDR1)       ;Read data byte
            RET                 ;Return character in A


;------------------------------------------------------------------------------------------------
; Character output
; Wait for device to be ready, then output character in C

list:       LD A,(iobyte)
            AND $C0
            CP $40
            JR NZ,conoutB
            JR conoutA

punch:      LD A,(iobyte)
            AND $20
            CP $20
            JR NZ,conoutB
            JR conoutA

conout:     LD A,(iobyte)
            AND $03
            CP $02
            JR Z,list           ; "BAT:" redirect
            CP $01
            JR NZ,conoutB

conoutA:    IN0  A,(STAT0)      ;Read serial port status register
            BIT  ST_TDRE,A      ;Transmit register empty?
            JR   Z,conoutA      ;No, so keep waiting
            OUT0 (TDR0), C      ;Write byte to serial port
            RET

conoutB:    IN0  A,(STAT1)      ;Read serial port status register
            BIT  ST_TDRE,A      ;Transmit register empty?
            JR   Z,conoutB      ;No, so keep waiting
            OUT0 (TDR1), C      ;Write byte to serial port
            RET


;------------------------------------------------------------------------------------------------
listst:     LD A,$FF            ; Return list status of 0xFF (ready).
            RET

;================================================================================================
; Disk processing entry points
;================================================================================================

seldsk:
            LD HL,$0000
            LD A,C
; <JL> Added IFDEF/ELSE block to select 64/128 MB
#IFDEF      SIZE64
            CP 8                ; 8 for 64MB disk, 16 for 128MB disk
#ELSE
            CP 16               ; 16 for 128MB disk, 8 for 64MB disk
#ENDIF
            jr C,chgdsk         ; if invalid drive will give BDOS error
            LD A,(userdrv)      ; so set the drive back to a:
            CP C                ; If the default disk is not the same as the
            RET NZ              ; selected drive then return, 
            XOR A               ; else reset default back to a:
            LD (userdrv),A      ; otherwise will be stuck in a loop
            LD (sekdsk),A
            ret

chgdsk:     LD  (sekdsk),A
            RLC A               ;*2
            RLC A               ;*4
            RLC A               ;*8
            RLC A               ;*16
            LD  HL,dpbase
            LD B,0
            LD c,A 
            ADD HL,BC

            RET

;------------------------------------------------------------------------------------------------
home:
            ld a,(hstwrt)       ;check for pending write
            or A
            jr nz,homed
            ld (hstact),A       ;clear host active flag
homed:
            LD  BC,0000h

;------------------------------------------------------------------------------------------------
settrk:     LD  (sektrk),BC     ; Set track passed from BDOS in register BC.
            RET

;------------------------------------------------------------------------------------------------
setsec:     LD  (seksec),BC     ; Set sector passed from BDOS in register BC.
            RET

;------------------------------------------------------------------------------------------------
setdma:     LD  (dmaAddr),BC    ; Set DMA ADDress given by registers BC.
            RET

;------------------------------------------------------------------------------------------------
sectran:    PUSH  BC
            POP  HL
            RET

;------------------------------------------------------------------------------------------------
read:
            ;read the selected CP/M sector
            xor A
            ld (unacnt),A
            ld A,1
            ld (readop),A       ;read operation
            ld (rsflag),A       ;must read data
            ld A,wrual
            ld (wrtype),A       ;treat as unalloc
            jp rwoper           ;to perform the read


;------------------------------------------------------------------------------------------------
write:
            ;write the selected CP/M sector
            xor A               ;0 to accumulator
            ld (readop),A       ;not a read operation
            ld A,C              ;write type in c
            ld (wrtype),A
            cp wrual            ;write unallocated?
            jr nz,chkuna        ;check for unalloc
;
;                               write to unallocated, set parameters
            ld A,blksiz/128     ;next unalloc recs
            ld (unacnt),A
            ld A,(sekdsk)       ;disk to seek
            ld (unadsk),A       ;unadsk = sekdsk
            ld HL,(sektrk)
            ld (unatrk),HL      ;unatrk = sectrk
            ld A,(seksec)
            ld (unasec),A       ;unasec = seksec
;
chkuna:
;                               check for write to unallocated sector
            ld A,(unacnt)       ;any unalloc remain?
            or A 
            jr z,alloc          ;skip if not
;
;                               more unallocated records remain
            dec A               ;unacnt = unacnt-1
            ld (unacnt),A
            ld A,(sekdsk)       ;same disk?
            ld HL,unadsk
            cp (HL)             ;sekdsk = unadsk?
            jp nz,alloc         ;skip if not
;
;                               disks are the same
            ld HL,unatrk
            call sektrkcmp      ;sektrk = unatrk?
            jp nz,alloc         ;skip if not
;
;                               tracks are the same
            ld A,(seksec)       ;same sector?
            ld HL,unasec
            cp (HL)             ;seksec = unasec?
            jp nz,alloc         ;skip if not
;
;                               match, move to next sector for future ref
            inc (HL)            ;unasec = unasec+1
            ld A,(HL)           ;end of track?
            cp cpmspt           ;count CP/M sectors
            jr c,noovf          ;skip if no overflow
;
;                               overflow to next track
            ld (HL),0           ;unasec = 0
            ld HL,(unatrk)
            inc HL
            ld (unatrk),HL      ;unatrk = unatrk+1
;
noovf:
            ;match found, mark as unnecessary read
            xor a               ;0 to accumulator
            ld (rsflag),a       ;rsflag = 0
            jr rwoper           ;to perform the write
;
alloc:
            ;not an unallocated record, requires pre-read
            xor a               ;0 to accum
            ld (unacnt),a       ;unacnt = 0
            inc a               ;1 to accum
            ld (rsflag),a       ;rsflag = 1

;------------------------------------------------------------------------------------------------
rwoper:
            ;enter here to perform the read/write
            xor a               ;zero to accum
            ld (erflag),a       ;no errors (yet)
            ld a,(seksec)       ;compute host sector
            or a                ;carry = 0
            rra                 ;shift right
            or a                ;carry = 0
            rra                 ;shift right
            ld (sekhst),a       ;host sector to seek
;
;                               active host sector?
            ld hl,hstact        ;host active flag
            ld a,(hl)
            ld (hl),1           ;always becomes 1
            or a                ;was it already?
            jr z,filhst         ;fill host if not
;
;                               host buffer active, same as seek buffer?
            ld a,(sekdsk)
            ld hl,hstdsk        ;same disk?
            cp (hl)             ;sekdsk = hstdsk?
            jr nz,nomatch
;
;                               same disk, same track?
            ld hl,hsttrk
            call sektrkcmp      ;sektrk = hsttrk?
            jr nz,nomatch
;
;                               same disk, same track, same buffer?
            ld a,(sekhst)
            ld hl,hstsec        ;sekhst = hstsec?
            cp (hl)
            jr z,match          ;skip if match
;
nomatch:
            ;proper disk, but not correct sector
            ld a,(hstwrt)       ;host written?
            or a
            call nz,writehst    ;clear host buff
;
filhst:
            ;may have to fill the host buffer
            ld a,(sekdsk)
            ld (hstdsk),a
            ld hl,(sektrk)
            ld (hsttrk),hl
            ld a,(sekhst)
            ld (hstsec),a
            ld a,(rsflag)       ;need to read?
            or a
            call nz,readhst     ;yes, if 1
            xor a               ;0 to accum
            ld (hstwrt),a       ;no pending write
;
match:
            ;copy data to or from buffer
            ld a,(seksec)       ;mask buffer number
            and secmsk          ;least signif bits
            ld l,a              ;ready to shift
            ld h,0              ;double count
            add hl,hl
            add hl,hl
            add hl,hl
            add hl,hl
            add hl,hl
            add hl,hl
            add hl,hl
;                               hl has relative host buffer address
            ld de,hstbuf
            add hl,de           ;hl = host address
            ex de,hl            ;now in DE
            ld hl,(dmaAddr)     ;get/put CP/M data
            ld c,128            ;length of move
            ld a,(readop)       ;which way?
            or a
            jr nz,rwmove        ;skip if read
;
;           write operation, mark and switch direction
            ld a,1
            ld (hstwrt),a       ;hstwrt = 1
            ex de,hl            ;source/dest swap
;
rwmove:
            ;C initially 128, DE is source, HL is dest
            ld a,(de)           ;source character
            inc de
            ld (hl),a           ;to dest
            inc hl
            dec c               ;loop 128 times
            jr nz,rwmove
;
;                               data has been moved to/from host buffer
            ld a,(wrtype)       ;write type
            cp wrdir            ;to directory?
            ld a,(erflag)       ;in case of errors
            ret nz              ;no further processing
;
;                               clear host buffer for directory write
            or a                ;errors?
            ret nz              ;skip if so
            xor a               ;0 to accum
            ld (hstwrt),a       ;buffer written
            call writehst
            ld a,(erflag)
            ret

;------------------------------------------------------------------------------------------------
;Utility subroutine for 16-bit compare
sektrkcmp:
            ;HL = .unatrk or .hsttrk, compare with sektrk
            ex de,hl
            ld hl,sektrk
            ld a,(de)           ;low byte compare
            cp (HL)             ;same?
            ret nz              ;return if not
;                               low bytes equal, test high 1s
            inc de
            inc hl
            ld a,(de)
            cp (hl)             ;sets flags
            ret

;================================================================================================
; Convert track/head/sector into LBA for physical access to the disk
;================================================================================================
setLBAaddr:
            LD HL,(hsttrk)
            RLC L
            RLC L
            RLC L
            RLC L
            RLC L
            LD A,L
            AND 0E0H
            LD L,A
            LD A,(hstsec)
            ADD A,L
            LD (lba0),A

            LD HL,(hsttrk)
            RRC L
            RRC L
            RRC L
            LD A,L
            AND 01FH
            LD L,A
            RLC H
            RLC H
            RLC H
            RLC H
            RLC H
            LD A,H
            AND 020H
            LD H,A
            LD A,(hstdsk)
            RLC a
            RLC a
            RLC a
            RLC a
            RLC a
            RLC a
            AND 0C0H
            ADD A,H
            ADD A,L
            LD (lba1),A
            

            LD A,(hstdsk)
            RRC A
            RRC A
            AND 03H
            LD (lba2),A

; LBA Mode using drive 0 = E0
            LD a,0E0H
            LD (lba3),A


            LD A,(lba0)
            OUT  (CF_LBA0),A

            LD A,(lba1)
            OUT  (CF_LBA1),A

            LD A,(lba2)
            OUT  (CF_LBA2),A

            LD A,(lba3)
            OUT  (CF_LBA3),A

            LD  A,1
            OUT  (CF_SECCOUNT),A

            RET    

;================================================================================================
; Read physical sector from host
;================================================================================================

readhst:
            PUSH  AF
            PUSH  BC
            PUSH  HL

            CALL  cfWait

            CALL  setLBAaddr

            LD  A,CF_READ_SEC
            OUT  (CF_COMMAND),A

            CALL  cfWait

            LD  c,4
            LD  HL,hstbuf
rd4secs:
            LD  b,128
rdByte:
            in  A,(CF_DATA)
            LD  (HL),A
            iNC  HL
            dec  b
            JR  NZ, rdByte
            dec  c
            JR  NZ,rd4secs

            POP  HL
            POP  BC
            POP  AF

            XOR  a
            ld (erflag),a
            RET

;================================================================================================
; Write physical sector to host
;================================================================================================

writehst:
            PUSH  AF
            PUSH  BC
            PUSH  HL


            CALL  cfWait

            CALL  setLBAaddr

            LD  A,CF_WRITE_SEC
            OUT  (CF_COMMAND),A

            CALL  cfWait

            LD  c,4
            LD  HL,hstbuf
wr4secs:
            LD  b,128
wrByte:     LD  A,(HL)
            OUT  (CF_DATA),A
            iNC  HL
            dec  b
            JR  NZ, wrByte

            dec  c
            JR  NZ,wr4secs

            POP  HL
            POP  BC
            POP  AF

            XOR  a
            ld (erflag),a
            RET

;================================================================================================
; Wait for disk to be ready (busy=0,ready=1)
;================================================================================================
#IFDEF      GrantsOriginal
cfWait:
            PUSH  AF
cfWait1:
            in  A,(CF_STATUS)
            AND  080H
            cp  080H
            JR Z,cfWait1
            POP  AF
            RET
#ELSE
cfWait:     PUSH AF
@TstBusy:   IN   A,(CF_STATUS)  ;Read status register
            BIT  7,A            ;Test Busy flag
            JR   NZ,@TstBusy    ;High so busy
@TstReady:  IN   A,(CF_STATUS)  ;Read status register
            BIT  6,A            ;Test Ready flag
            JR   Z,@TstBusy     ;Low so not ready
            POP  AF
            RET
#ENDIF

;================================================================================================
; Utilities
;================================================================================================

printInline:
            EX  (SP),HL         ; PUSH HL and put RET ADDress into HL
            PUSH  AF
            PUSH  BC
nextILChar: LD  A,(HL)
            CP 0
            JR Z,endOfPrint
            LD   C,A
            CALL  conout        ; Print to TTY
            iNC  HL
            JR nextILChar
endOfPrint: INC  HL             ; Get past "null" terminator
            POP  BC
            POP  AF
            EX  (SP),HL         ; PUSH new RET ADDress on stack and restore HL
            RET

;================================================================================================
; Data storage
;================================================================================================

dirbuf:     .ds 128             ;scratch directory area
alv00:      .ds 257             ;allocation vector 0
alv01:      .ds 257             ;allocation vector 1
alv02:      .ds 257             ;allocation vector 2
alv03:      .ds 257             ;allocation vector 3
alv04:      .ds 257             ;allocation vector 4
alv05:      .ds 257             ;allocation vector 5
alv06:      .ds 257             ;allocation vector 6
alv07:      .ds 257             ;allocation vector 7
; <JL> Added IFDEF block to select 64/128 MB
#IFDEF      SIZE128
alv08:      .ds 257             ;allocation vector 8
alv09:      .ds 257             ;allocation vector 9
alv10:      .ds 257             ;allocation vector 10
alv11:      .ds 257             ;allocation vector 11
alv12:      .ds 257             ;allocation vector 12
alv13:      .ds 257             ;allocation vector 13
alv14:      .ds 257             ;allocation vector 14
alv15:      .ds 257             ;allocation vector 15
#ENDIF

lba0        .DB 00h
lba1        .DB 00h
lba2        .DB 00h
lba3        .DB 00h

            .DS 020h            ; Start of BIOS stack area.
biosstack:  .EQU $

sekdsk:     .ds 1               ;seek disk number
sektrk:     .ds 2               ;seek track number
seksec:     .ds 2               ;seek sector number
;
hstdsk:     .ds 1               ;host disk number
hsttrk:     .ds 2               ;host track number
hstsec:     .ds 1               ;host sector number
;
sekhst:     .ds 1               ;seek shr secshf
hstact:     .ds 1               ;host active flag
hstwrt:     .ds 1               ;host written flag
;
unacnt:     .ds 1               ;unalloc rec cnt
unadsk:     .ds 1               ;last unalloc disk
unatrk:     .ds 2               ;last unalloc track
unasec:     .ds 1               ;last unalloc sector
;
erflag:     .ds 1               ;error reporting
rsflag:     .ds 1               ;read sector flag
readop:     .ds 1               ;1 if read operation
wrtype:     .ds 1               ;write operation type
dmaAddr:    .ds 2               ;last dma address
hstbuf:     .ds 512             ;host buffer

hstBufEnd:  .EQU $


;serialVarsEnd:                 .EQU $


biosEnd:    .EQU $

; Disable the ROM, pop the active IO port from the stack (supplied by monitor),
; then start CP/M
popAndRun:
            LD A,$01 
            OUT ($38),A

            POP AF
            CP $01
            JR Z,consoleAtB
            LD A,$01            ;(List is TTY:, Punch is TTY:, Reader is TTY:, Console is CRT:)
            JR setIOByte
consoleAtB: LD A,$00            ;(List is TTY:, Punch is TTY:, Reader is TTY:, Console is TTY:)
setIOByte:  LD (iobyte),A

            JP bios


            ;.org 0FFE0H
            ;.dw serialInt


;=================================================================================
; Relocate TPA area from 4100 to 0100 then start CP/M
; Used to manually transfer a loaded program after CP/M was previously loaded
;=================================================================================

            .org 0FFE8H
            LD A,$01
            OUT ($38),A

            LD HL,04100H
            LD DE,00100H
            LD BC,08F00H
            LDIR
            JP bios

;=================================================================================
; Normal start CP/M vector
;=================================================================================

            .ORG 0FFFEH
            .dw popAndRun

            .END













